<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://thehp.in
 * @since      1.0.0
 *
 * @package    Global_Dns
 * @subpackage Global_Dns/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Global_Dns
 * @subpackage Global_Dns/public
 */
class Global_Dns_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->servers = get_option($this->plugin_name . '-servers');
		if (!is_array($this->servers)) {
			$this->servers = array();
		}
		foreach ($this->servers as $i => $server) {
			if (isset($this->servers[$i]['long'])) {
				$this->servers[$i]['longitude'] = $this->servers[$i]['long'];
			}
			if (isset($this->servers[$i]['lat'])) {
				$this->servers[$i]['latitude'] = $this->servers[$i]['lat'];
			}
			if (isset($this->servers[$i]['url'])) {
				$this->servers[$i]['url_ip'] = $this->servers[$i]['url'];
			}
		}
		$this->settings = get_option($this->plugin_name . '-settings');
		if (!is_array($this->settings)) {
			$this->settings = get_option($this->plugin_name);
		}
		$this->settings['siteurl'] = get_option('siteurl');
		$this->settings['pluginurl'] = plugin_dir_url(__FILE__);
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Global_Dns_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Global_Dns_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/global-dns-public.css', array(), $this->version, 'all');
		wp_enqueue_style('jquery-jvectormap', plugins_url($this->plugin_name) . '/vendor/jvectormap/jquery-jvectormap-2.0.5.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Global_Dns_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Global_Dns_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/global-dns-public.min.js', array(), $this->version, false);
		wp_enqueue_script($this->plugin_name . '-jquery', plugin_dir_url(__FILE__) . 'js/global-dns-public-jquery.min.js', array('jquery'), $this->version, false);
		wp_enqueue_script('jquery-jvectormap', plugins_url($this->plugin_name) . '/vendor/jvectormap/jquery-jvectormap-2.0.5.min.js', array('jquery'), $this->version, false);
		wp_enqueue_script('jquery-jvectormap-world-mill', plugins_url($this->plugin_name) . '/vendor/jvectormap/jquery-jvectormap-world-mill.js', array('jquery'), $this->version, false);
	}

	/**
	 * Register JS and CSS Assets
	 * 
	 * @since	1.0.0
	 */
	public function enqueue_shortcode_styles_scripts() {
		$this->enqueue_scripts();
		$this->enqueue_styles();
	}

	/**
	 * Fetch the DNS records from external server OR DNS Server
	 * 
	 * @since	1.0.0
	 */
	public function global_dns_server_api($data) {
		if (!isset($this->servers[$data['id']])) {
			return "";
		}
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if (strpos($this->servers[$data['id']]['url_ip'], 'http') !== false) {
			$response = wp_remote_get(
				$this->servers[$data['id']]['url_ip'],
				array(
					'method' => 'POST',
					'body' => array(
						'url' => base64_decode($data['domain']),
						'type' => $data['type'],
						'secret' => $this->settings['secret']
					)
				)
			);
			if ($response['response']['code'] == 401) {
				return new WP_Error('invalid_secret', 'Server Authentication Error', array('status' => 401));
			}
			return wp_remote_retrieve_body($response);
		} else {
			if ($data['type'] == "PTR") {
				return gethostbyaddr(base64_decode($data['domain']));
			}
			return $this->getRecords(base64_decode($data['domain']), $data['type'], $this->servers[$data['id']]['url_ip']);
		}
	}

	/**
	 * Creating REST API for WordPress
	 * 
	 * @since	1.0.0
	 */
	public function global_dns_server_register_rest_api() {
		register_rest_route('global-dns/v1', '/(?P<id>\d+)/(?P<type>[A-Z]+)/(?P<domain>[A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_api'),
			'args' => array(
				'id' => array(
					'validate_callback' => function ($param, $request, $key) {
						return is_numeric($param);
					}
				),
			),
		));
		register_rest_route('global-dns/v1', '/whois/(?P<domain>[A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_whois_api'),
		));
		register_rest_route('global-dns/v1', '/ip-lookup/(?P<ip>[\.A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_ip_lookup_api'),
		));
		register_rest_route('global-dns/v1', '/domain/ip/(?P<domain>[\.A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_domain_ip_api'),
		));
		register_rest_route('global-dns/v1', '/blacklist/(?P<ip>[\.A-Za-z0-9=]+)/(?P<blacklist>[\.A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_blacklist_api'),
		));
		register_rest_route('global-dns/v1', '/dmarc/(?P<domain>[A-Za-z0-9=]+)', array(
			'methods' => 'POST',
			'callback' => array($this, 'global_dns_server_dmarc_api'),
		));
	}

	/**
	 * WHOIS API
	 * 
	 * @since	2.0.0
	 */
	public function global_dns_server_whois_api($data) {
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if ($data['domain']) {
			return $this->getWhois(base64_decode($data['domain']));
		}
		return "NA";
	}

	/**
	 * WHOIS API
	 * 
	 * @since	2.0.0
	 */
	public function global_dns_server_ip_lookup_api($data) {
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if ($data['ip']) {
			$url = "http://ip-api.com/json/" . $data['ip'] . "?fields=8454143";
			$response = wp_remote_get($url);
			return [
				'ignore' => ['status'],
				'headers' => [
					'country' => 'Country',
					'zip' => 'ZIP',
					'countryCode' => 'Country Code',
					'region' => 'Region Code',
					'regionName' => 'Region',
					'lat' => 'Latitude',
					'lon' => 'Longitude',
					'isp' => 'ISP',
					'as' => 'AS Name',
					'org' => 'Organisation',
					'timezone' => 'Timezone',
					'reverse' => 'Hostname',
					'city' => 'City',
					'currency' => 'Currency',
					'query' => 'IP'
				],
				'result' => json_decode($response['body'], true)
			];
		}
		return "NA";
	}

	/**
	 * API to get IP from Hostname - Used for Blacklist Checker
	 * 
	 * @since	2.6.0
	 */
	public function global_dns_server_domain_ip_api($data) {
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if ($data['domain']) {
			return gethostbyname(base64_decode($data['domain']));
		}
		return null;
	}

	/**
	 * Blacklist API
	 * 
	 * @since	2.6.0
	 */
	public function global_dns_server_blacklist_api($data) {
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if ($data['ip'] && $data['blacklist']) {
			$array = explode('.', $data['ip']);
			$array = array_reverse($array);
			$ip = implode('.', $array);
			$blacklist = $data['blacklist'];
			if (dns_check_record($ip . '.' . $blacklist, 'A')) {
				return 'listed';
			}
			return 'not-listed';
		}
		return "NA";
	}

	/**
	 * DMARC API
	 * 
	 * @since	2.7.0
	 */
	public function global_dns_server_dmarc_api($data) {
		if (!check_ajax_referer('wp_rest', '_wpnonce', false)) {
			return new WP_Error('invalid_nonce', 'Bad Request', array('status' => 400));
		}
		if ($data['domain']) {
			$domain = base64_decode($data['domain']);
			$record = dns_get_record('_dmarc.' . $domain, DNS_TXT);
			if (isset($record[0]) && isset($record[0]['txt'])) {
				return $record[0]['txt'];
			}
		}
		return "";
	}

	/**
	 * Registering the shortcodes
	 * 
	 * @since	1.0.0
	 */
	public function global_dns_shortcodes() {
		add_shortcode('global_dns_input', array($this, 'global_dns_shortcode_input'));
		add_shortcode('global_dns_list', array($this, 'global_dns_shortcode_list'));
		add_shortcode('global_dns_map', array($this, 'global_dns_shortcode_map'));
		add_shortcode('global_dns_whois_input', array($this, 'global_dns_shortcode_whois_input'));
		add_shortcode('global_dns_whois', array($this, 'global_dns_shortcode_whois'));
		add_shortcode('global_dns_ip_lookup_input', array($this, 'global_dns_shortcode_ip_lookup_input'));
		add_shortcode('global_dns_ip_lookup', array($this, 'global_dns_shortcode_ip_lookup'));
		add_shortcode('global_user_ip', array($this, 'global_dns_user_ip'));
		add_shortcode('global_dns_blacklist_input', array($this, 'global_dns_shortcode_blacklist_input'));
		add_shortcode('global_dns_blacklist', array($this, 'global_dns_shortcode_blacklist'));
		add_shortcode('global_dns_dmarc_input', array($this, 'global_dns_shortcode_dmarc_input'));
		add_shortcode('global_dns_dmarc', array($this, 'global_dns_shortcode_dmarc'));
		add_shortcode('global_dns_email_headers_input', array($this, 'global_dns_shortcode_email_headers_input'));
		add_shortcode('global_dns_email_headers', array($this, 'global_dns_shortcode_email_headers'));
	}

	/**
	 * Defining shortcode for INPUT field
	 * 
	 * @since 	1.0.0
	 */
	public function global_dns_shortcode_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$original_servers = $this->servers;
		$servers = array();
		for ($i = 0; $i < count($original_servers); $i++) {
			if ($original_servers[$i]['name']) {
				array_push($servers, array(
					'name' => $original_servers[$i]['name'],
					'long' => $original_servers[$i]['longitude'],
					'lat' => $original_servers[$i]['latitude'],
					'country' => $original_servers[$i]['country'],
					'result' => ''
				));
			}
		}
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/** 
	 * Defining shortcode for LIST of servers
	 * 
	 * @since 	1.0.0
	 */
	public function global_dns_shortcode_list($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$servers = $this->servers;
		$pluginurl = plugin_dir_url(__FILE__);
		include_once('partials/global-dns-shortcode-list.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for MAP of server location
	 * 
	 * @since	1.0.0
	 */
	public function global_dns_shortcode_map($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/global-dns-shortcode-map.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for WHOIS INPUT 
	 * 
	 * @since	2.0.0
	 */
	public function global_dns_shortcode_whois_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/whois/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for WHOIS Result 
	 * 
	 * @since	2.0.0
	 */
	public function global_dns_shortcode_whois($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/whois/global-dns-shortcode-whois.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for IP Lookup Input
	 * 
	 * @since	2.5.0
	 */
	public function global_dns_shortcode_ip_lookup_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/ip_lookup/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for IP Lookup Result
	 * 
	 * @since	2.5.0
	 */
	public function global_dns_shortcode_ip_lookup($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/ip_lookup/global-dns-shortcode-ip-lookup.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for Blacklist Input
	 * 
	 * @since	2.6.0
	 */
	public function global_dns_shortcode_blacklist_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/blacklist/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for Blacklist Results
	 * 
	 * @since	2.6.0
	 */
	public function global_dns_shortcode_blacklist($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/blacklist/global-dns-shortcode-blacklist.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for DMARC INPUT 
	 * 
	 * @since	2.7.0
	 */
	public function global_dns_shortcode_dmarc_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/dmarc/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for DMARC Result 
	 * 
	 * @since	2.7.0
	 */
	public function global_dns_shortcode_dmarc($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/dmarc/global-dns-shortcode-dmarc.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for DMARC INPUT 
	 * 
	 * @since	2.8.0
	 */
	public function global_dns_shortcode_email_headers_input($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		$settings = $this->settings;
		unset($settings['secret']);
		include_once('partials/email_headers/global-dns-shortcode-input.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Defining shortcode for DMARC Result 
	 * 
	 * @since	2.8.0
	 */
	public function global_dns_shortcode_email_headers($attr, $content = null) {
		$this->enqueue_shortcode_styles_scripts();
		ob_start();
		include_once('partials/email_headers/global-dns-shortcode-email-headers.php');
		$content = ob_get_clean();
		return $content;
	}

	/**
	 * Shortcode for User IP
	 * 
	 * @since	2.4.0
	 */
	public function global_dns_user_ip($attr, $content = null) {
		$ipaddress = '';
		if (getenv('HTTP_CLIENT_IP'))
			$ipaddress = getenv('HTTP_CLIENT_IP');
		else if (getenv('HTTP_X_FORWARDED_FOR'))
			$ipaddress = getenv('HTTP_X_FORWARDED_FOR');
		else if (getenv('HTTP_X_FORWARDED'))
			$ipaddress = getenv('HTTP_X_FORWARDED');
		else if (getenv('HTTP_FORWARDED_FOR'))
			$ipaddress = getenv('HTTP_FORWARDED_FOR');
		else if (getenv('HTTP_FORWARDED'))
			$ipaddress = getenv('HTTP_FORWARDED');
		else if (getenv('REMOTE_ADDR'))
			$ipaddress = getenv('REMOTE_ADDR');
		else
			$ipaddress = 'UNKNOWN';
		return $ipaddress;
	}

	/**
	 * Fetching the DNS records from DNS server
	 * 
	 * @since	1.1.0
	 */
	private function getRecords($domain, $type, $dns) {
		include('vendor/DigParser/DigParser.php');
		try {
			$response = shell_exec('dig @' . $dns . ' ' . $domain . ' ' . $type);
			if ($response == NULL) {
				return new WP_Error('dns_fetch_failed', 'shell_exec is disabled or non functional', array('status' => 400));
			}
			$parser = new DigParser();
			$data = $parser->parse($response);
			$return = '';
			if (is_array($data->answer)) {
				foreach ($data->answer as $key => $answer) {
					$return .= ($key ? '<br>' : '') . trim($answer->data, '.');
				}
			}
			return $return;
		} catch (Exception $e) {
			return new WP_Error('dns_fetch_failed', $e->getMessage());
		}
	}

	/**
	 * Fetching WHOIS 
	 * 
	 * @since	2.0.0
	 */
	private function getWhois($domain) {
		include('vendor/Whois/Whois.php');
		try {
			$whois = new Whois($domain);
			return $whois->info();
		} catch (Exception $e) {
			return new WP_Error('whois_fetch_failed', $e->getMessage());
		}
	}
}
